unit IWCompListBoxWAP;

interface

uses
  {$IFDEF VSNET}
  System.ComponentModel, System.Drawing,
  IWNetClasses,
  {$ELSE}
  Classes,
    {$IFDEF Linux}
    IWCLXComponent, IWCLXClasses,
    {$ELSE}
    IWVCLComponent, IWVCLClasses, Controls,
    {$ENDIF}
  {$ENDIF}                       
  IWControlWAP, IWBaseWAPInterfaces, IWBaseControl, IWBaseRenderContext, IWWapTag,
  IWBaseComponent, IWBaseInterfaces, IWMarkupLanguageTag, IWTypes, IWBaseWapControl,
  IWWAPRenderContext;

type
  TSelectControlType = (sctListbox, sctCombobox, sctCheckBox);

  TIWCustomListComboWAP = class(TIWCustomControlWAP, IIWTabOrder)
  private
    FItems: TIWStringList;

    procedure SetItemIndex(AIndex: Integer);
    procedure SetItems(const AValue: TIWStringList);
    procedure OnItemsChange(ASender: TObject);
  protected
    FItemIndex: Integer;
    FItemsHaveValues: Boolean;
    FNoSelectionText: string;
    FRequireSelection: Boolean;

    procedure InitItems; virtual;

    function getText: TCaption; override;
    procedure SetItemsHaveValues(const Value: Boolean);
    procedure InitControl; override;
    procedure SetRequireSelection(const AValue: Boolean); virtual;
    function GetItems: TIWStringList;
  {$IFDEF CLR}
  strict protected
  {$ELSE}
  protected
  {$ENDIF}
    procedure Dispose(ADispose: Boolean); override;
    {$IFDEF VSNET}
    [DesignerSerializationVisibility(DesignerSerializationVisibility.Content)]
    {$ENDIF}
    property Items: TIWStringList read GetItems write SetItems;

    {$IFDEF VSNET}
    [Bindable(true)]
    {$ENDIF}    
    property ItemIndex: Integer read FItemIndex write SetItemIndex;
  public
    {RenderWML creates only WML code. XHTML is generated separately in RenderWAP}
    function RenderWML(AContext: TIWBaseWapComponentContext;
                       ASelectControlType: TSelectControlType): TIWWapTag;
    property Text: TCaption read getText write SetText;
  published
    property NoSelectionText: string read FNoSelectionText write FNoSelectionText;
    property TabOrder;
    property RequireSelection: Boolean read FRequireSelection write SetRequireSelection;
    property Caption;
    property ItemsHaveValues: Boolean read FItemsHaveValues write SetItemsHaveValues;
  end;

  TIWCustomComboBoxWAP = class(TIWCustomListComboWAP, IIWInputControl)
  protected
    procedure InitControl; override;
    procedure SetValue(const AValue: string); virtual;
  public
    // Needed for PaintHandlers
    function RenderWap(AContext: TIWBaseWapComponentContext): TIWWapTag; override;
    property Items;
    property ItemIndex;
  published
    property Editable default true;
    property TabOrder;
  end;

  TIWCustomListBoxWAP = class(TIWCustomListComboWAP, IIWInputControl)
  protected
    FSelectedList: TList;
    FTempList : TIWStringList;
    FMultiSelect: Boolean;
    function GetSelected(AIndex: integer): boolean;
    procedure SetSelected(AIndex: integer; const AValue: boolean);
    procedure InitItems; override;
    procedure InitControl; override;
    procedure DoItemsChange(Sender: TObject);
    procedure DoItemsChanging(Sender: TObject);
    procedure SetMultiSelect(const AValue: Boolean);
    procedure SetValue(const AValue: string); virtual;
  {$IFDEF CLR}
  strict protected
  {$ELSE}
  protected
  {$ENDIF}
    procedure Dispose(ADispose: Boolean); override;
  public
    function RenderWap(AContext: TIWBaseWapComponentContext): TIWWapTag; override;
    procedure ResetSelection;
    property Items;
    property ItemIndex;
    property Selected[AIndex: Integer]: Boolean read GetSelected write SetSelected;
    property MultiSelect: boolean read FMultiSelect write SetMultiSelect;
  published
    property Editable default true;
    property TabOrder;
  end;

  {$IFDEF VSNET}
  {$R icons\Atozed.Intraweb.TIWComboBoxWAP.bmp}
  TIWComboBoxWAP = class;
  [ToolboxItem(true), ToolboxBitmap(typeof(TIWComboBoxWAP), 'TIWComboBoxWAP.bmp')]
  {$ENDIF}
  TIWComboBoxWAP = class(TIWCustomComboBoxWAP)
  published
    property Items;
    property ItemIndex;
  end;

  {$IFDEF VSNET}
  {$R icons\Atozed.Intraweb.TIWListBoxWAP.bmp}
  TIWListBoxWAP = class;
  [ToolboxItem(true), ToolboxBitmap(typeof(TIWListBoxWAP), 'TIWListBoxWAP.bmp')]
  {$ENDIF}
  TIWListBoxWAP = class(TIWCustomListBoxWAP)
  published
    property Items;
    property ItemIndex;
    property MultiSelect; 
  end;


implementation

uses
  SysUtils, IWCompLabelWAP, SWStrings, SWSystem {$IFDEF VER130}, Windows{$ENDIF};


{ TIWCustomListComboWAP }

procedure TIWCustomListComboWAP.InitControl;
begin
  inherited;
  FItemIndex := -1;
  Editable := True;
  RequireSelection := True;
  FNoSelectionText := '-- No Selection --';
end;

procedure TIWCustomListComboWAP.Dispose(ADispose: Boolean);
begin
  FreeAndNil(FItems);
  inherited;
end;

function TIWCustomListComboWAP.RenderWML(AContext: TIWBaseWapComponentContext;
                                         ASelectControlType: TSelectControlType): TIWWapTag;
var
  i: Integer;
  LText: string;
  LValue: string;
begin
  Result := nil;
  if Editable then begin
    Result := TIWWAPTag.CreateWAPTag('select'); try
      Result.AddStringParam('name', Name);

      Result.AddStringParam('title', Caption{Todo published?});
      case ASelectControlType of
        sctListbox,sctCheckBox: begin
          Result.AddStringParam('multiple', 'true');
        end;
      end;

      if ItemIndex >= 0 then begin
        Result.AddIntegerParam('value', ItemIndex);
      end;

      Result.AddIntegerParam('tabindex', TabOrder);

      if Items.Count > 0 then begin
        if ((FItemIndex = -1) or (RequireSelection = False)) and (ASelectControlType <> sctCheckBox) then begin
          with Result.Contents.AddTag('option') do begin
            AddIntegerParam('value', -1);
            Contents.AddText(FNoSelectionText);
          end;
        end;
        for i := 0 to Items.Count - 1 do begin
          LText := Items.Strings[i];
          LValue := '';

          if ItemsHaveValues then begin
            LValue := LText;
            LText := Fetch(LValue, '=');
          end else begin
            LValue := IntToStr(i);
          end;
          LValue := IntToStr(i);
          with Result.Contents.AddTag('option') do begin
            AddStringParam('value', LValue);
            Contents.AddText(LText);
          end;
        end;
      end
      else begin
        Result.Contents.AddText('');
      end;
    except FreeAndNil(Result); raise; end;
  end else begin
    with TIWLabelWAP.Create{$IFNDEF VSNET}(Self){$ENDIF} do try
      Name := Self.Name;
      ExtraTagParams.Assign(Self.ExtraTagParams);
      // Font.Assign(Self.Font);
      Width := Self.Width;
      Height := Self.Height;
      Caption := '';
      if (Self.Items.Count > 0) and (Self.ItemIndex > -1) then begin
        Caption := Self.Items.Strings[Self.ItemIndex];
      end;
      Result := RenderWAP(AContext);
    finally Free; end;
  end;
end;

procedure TIWCustomListComboWAP.SetItemIndex(AIndex: Integer);
begin
  {$IFDEF VSNET}
  if not assigned(FItems) then
  {$ELSE}
  if IsLoading then
  {$ENDIF}
  begin
    // Set no matter what, it might be set (and usually is) before the items are loaded
    FItemIndex := AIndex;
  end else if AIndex < Items.Count then begin
    FItemIndex := AIndex;
    Invalidate;
  end;
end;

procedure TIWCustomListComboWAP.SetItems(const AValue: TIWStringList);
begin
  if (AValue.Count = 0) or (AValue.Count < Items.Count) then begin
    FItemIndex := -1;
  end;
  Items.Assign(AValue);
  Invalidate;
end;

procedure TIWCustomListComboWAP.SetRequireSelection(const AValue: Boolean);
begin
  FRequireSelection := AValue;
end;

procedure TIWCustomListComboWAP.SetItemsHaveValues(const Value: Boolean);
begin
  FItemsHaveValues := Value;
  Invalidate;
end;

procedure TIWCustomListComboWAP.OnItemsChange(ASender: TObject);
begin
  Invalidate;
end;

procedure TIWCustomListComboWAP.InitItems;
begin
  FItems := TIWStringList.Create;
  FItems.OnChange := OnItemsChange;
end;

function TIWCustomListComboWAP.GetItems: TIWStringList;
begin
  if not Assigned(FItems) then begin
    InitItems;
  end;
  result := FItems;
end;

{ TIWCustomComboBoxWAP }

procedure TIWCustomComboBoxWAP.InitControl;
begin
  inherited;
  Height := 21;
  Width := 121;
end;

procedure TIWCustomComboBoxWAP.SetValue(const AValue: string);
var
  s: string;
  i: integer;
begin
  if RequiresUpdateNotification(Parent) then begin
    UpdateNotifiedInterface(Parent).NotifyUpdate(Self,AValue);
  end;
  s := AValue;
  if ItemsHaveValues then begin
    for i := 0 to Items.Count - 1 do begin
      if S = Items.Values[Items.Names[i]] then begin
        ItemIndex := i;
        break;
      end;
    end;
    if i = Items.Count then begin
      ItemIndex := -1;
      Invalidate;
    end;
  end else begin
    ItemIndex := StrToIntDef(Fetch(s, ','), -1);
  end;
  Invalidate;
end;

function TIWCustomComboBoxWAP.RenderWAP(AContext: TIWBaseWAPComponentContext): TIWWapTag;
var
  i: Integer;
  LText: string;
  LValue: string;
begin
  Result := nil;
  if AContext.PageContext.OutputTagType = wotWML then begin
    Result := RenderWML(AContext, sctCombobox);
  end else begin
    if true{Editable} {Todo: Why it's always not Editable?} then begin
      Result := TIWWapTag.CreateTag('select'); try
        Result.AddStringParam('name', Name);
        Result.AddIntegerParam('size', 1);
        if Items.Count > 0 then begin
          if ((FItemIndex = -1) or (RequireSelection = False)) then begin
            with Result.Contents.AddTag('option') do begin
              AddStringPAram('selected', 'selected');
              AddIntegerParam('value', -1);
              Contents.AddText(FNoSelectionText);
            end;
          end;
          for i := 0 to Items.Count - 1 do begin
            LText := Items.Strings[i];
            LValue := '';
            if ItemsHaveValues then begin
              LValue := LText;
              LText := Fetch(LValue, '=');
            end else begin
              LValue := IntToStr(i);
            end;
            with Result.Contents.AddTag('option') do begin
              if ItemIndex = i then begin
                AddStringPAram('selected', 'selected');
              end;
              AddStringParam('value', LValue);
              Contents.AddText(TextToWML(LText));
            end;
          end;
        end
        else begin
          Result.Contents.AddText('');
        end;
      except FreeAndNil(Result); raise; end;
    end else begin
      with TIWLabelWAP.Create{$IFNDEF VSNET}(Self){$ENDIF} do try
        Name := Self.Name;
        ExtraTagParams.Assign(Self.ExtraTagParams);
        // Font.Assign(Self.Font);
        Width := Self.Width;
        Height := Self.Height;
        Caption := '';
       // RawText := FALSE;
        if (Self.Items.Count > 0) and (Self.ItemIndex > -1) then begin
          Caption := TextToWML(Self.Items.Strings[Self.ItemIndex]);
        end;
        Result := RenderWAP(AContext);
      finally Free; end;
    end;
  end;
end;


{ TIWCustomListBoxWAP }

procedure TIWCustomListBoxWAP.InitControl;
begin
  inherited;
  Height := 21;
  Width := 121;
  FSelectedList := TList.Create;
  FMultiSelect := true;

  FTempList := TIWStringList.Create;
end;

procedure TIWCustomListBoxWAP.Dispose(ADispose: Boolean);
begin
  FreeAndNil(FTempList);
  FreeAndNil(FSelectedList);
  inherited;
end;

procedure TIWCustomListboxWAP.DoItemsChange(Sender: TObject);
var
  f : integer;
  LTempSelected : TList;
begin
  if not(MultiSelect and (FSelectedList.Count > 0)) then
  begin
    Exit;
  end;

  LTempSelected := TList.Create;
  try
     for f := 0 to Pred(FTempList.Count) do
     begin
        if Items.IndexOf(FTempList.Strings[f]) >= 0 then
        begin
           if FSelectedList.IndexOf(TObject(f)) >= 0 then
           begin
              LTempSelected.Add(TObject(Items.IndexOf(FTempList.Strings[f])));
           end;
        end;
     end;
     FSelectedList.Clear;
     {$IFDEF VER130}
     FSelectedList.Capacity := LTempSelected.Count;//
     CopyMemory(FSelectedList.List, LTempSelected.List, SizeOf(LTempSelected.List^[0]) * LTempSelected.Count);
     {$ELSE}
     FSelectedList.Assign(LTempSelected);
     {$ENDIF}
  finally
     FreeAndNil(LTempSelected);
  end;
end;

procedure TIWCustomListboxWAP.DoItemsChanging(Sender: TObject);
begin
  if not(MultiSelect and (FSelectedList.Count > 0)) then
  begin
    Exit;
  end;

  FTempList.Assign(Items);
end;

function TIWCustomListboxWAP.GetSelected(AIndex: integer): boolean;
begin
  if FMultiSelect then begin
    Result := FSelectedList.IndexOf(TObject(AIndex)) > -1;
  end else begin
    Result := AIndex = ItemIndex;
  end;
end;

procedure TIWCustomListBoxWAP.SetValue(const AValue: string);
var
  s: string;
  i: integer;
begin
  if RequiresUpdateNotification(Parent) then begin
    UpdateNotifiedInterface(Parent).NotifyUpdate(Self,AValue);
  end;
  s := AValue;

  if ItemsHaveValues then begin
    for i := 0 to Items.Count - 1 do begin
      if S = Items.Values[Items.Names[i]] then begin
        ItemIndex := i;
        break;
      end;
    end;
    if i = Items.Count then begin
      ItemIndex := -1;
      Invalidate;
    end;
  end else begin
    ItemIndex := StrToIntDef(Fetch(s, ','), -1);
  end;
  Invalidate;
end;

procedure TIWCustomListboxWAP.ResetSelection;
begin
  FSelectedList.Clear;
  Invalidate;
end;

procedure TIWCustomListboxWAP.SetMultiSelect(const AValue: Boolean);
begin
  if AValue then begin
    FRequireSelection := False;
  end;
  FMultiSelect := AValue;
end;

procedure TIWCustomListboxWAP.SetSelected(AIndex: integer;
  const AValue: boolean);
begin
  if AValue then begin
    if not GetSelected(AIndex) then begin
      FSelectedList.Add(TObject(AIndex));
    end;
  end else begin
    FSelectedList.Remove(TObject(AIndex));
  end;
  Invalidate;
end;

function TIWCustomListBoxWAP.RenderWAP(AContext: TIWBaseWAPComponentContext): TIWWapTag;
var
  i: Integer;
  LText: string;
  LValue: string;
begin
  Result := nil;
  if AContext.PageContext.OutputTagType = wotWML then begin
    Result := RenderWML(AContext, sctListbox);
  end else begin
    if true{Editable} {Todo: Why it's always not Editable?} then begin
    {Todo: If the above todo gets resolved, then support for NonEditableAsLabel might be implemented}
      Result := TIWWAPTag.CreateTag('select'); try
        Result.AddStringParam('name', Name);
        Result.AddIntegerParam('size', Height div 16);
        Result.Add(iif(FMultiSelect, 'multiple'));
        if Items.Count > 0 then begin
          if ((FItemIndex = -1) or (RequireSelection = False))
           and (FMultiSelect = False) then begin
            with Result.Contents.AddTag('option') do begin
              AddStringParam('selected','selected');
              AddIntegerParam('value', -1);
              Contents.AddText(FNoSelectionText);
            end;
          end;
          for i := 0 to Items.Count - 1 do begin
            LText := Items.Strings[i];
            LValue := '';
            if ItemsHaveValues then begin
              LValue := LText;
              LText := Fetch(LValue, '=');
            end else begin
              LValue := IntToStr(i);
            end;
            with Result.Contents.AddTag('option') do begin
              if (
               (ItemIndex = i) and (FMultiSelect = False))
               or (Selected[i] and FMultiSelect)
               then begin
                AddStringPAram('selected', 'selected');
               end;
              AddStringParam('value', LValue);
              Contents.AddText(TextToWML(LText));
            end;
          end;
        end
        else begin
          Result.Contents.AddText('');
        end;
      except FreeAndNil(Result); raise; end;
    end else begin
      with TIWLabelWAP.Create{$IFNDEF VSNET}(Self){$ENDIF} do try
        Name := Self.Name;
        ExtraTagParams.Assign(Self.ExtraTagParams);
        // Font.Assign(Self.Font);
        Width := Self.Width;
        Height := Self.Height;
        Caption := '';
//        RawText := FALSE; // Otherwise font info won't be rendered
        if Self.ItemIndex > -1 then begin
          if ItemsHaveValues then begin
            Caption := TextToWML(Self.Items.Values[Self.Items.Names[Self.ItemIndex]]);
          end else begin
            Caption := TextToWML(Self.Items.Strings[Self.ItemIndex]);
          end;
        end;
        Result := RenderWAP(AContext);
      finally Free; end;
    end;
  end;
end;

procedure TIWCustomListBoxWAP.InitItems;
begin
  inherited;
  FItems.OnChange := DoItemsChange;
  FItems.OnChanging := DoItemsChanging;
end;

function TIWCustomListComboWAP.getText: TCaption;
begin
  if (ItemIndex > -1) and (ItemIndex < Items.Count) then begin
    if ItemsHaveValues then begin
      Result := Items.Values[Items.Names[ItemIndex]];
    end else begin
      Result := Items.Strings[ItemIndex];
    end;
  end else begin
    Result := '';
  end;
end;

end.
